/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

#pragma warning( disable: 4115 4514)
#include <windows.h>
#pragma warning( default: 4115 4514)

#include <typedefs.h>

#include "b_io.h"
#include "b_serial.h"
#include "lasterr.h"

/* --------------------------------------------------------------------------
 * STATICS
 * -------------------------------------------------------------------------- */

/******************************************************************************/
static b_errtype BestClearCommError(b_portnumtype OsHandle)
{
  unsigned long lErrors = 0UL;
  int MaxTries = 10;
  COMSTAT ComStat;

  /* Note; Do not use FlushFileBuffers(),
   * it will transmit the rest of the output buffer !
   */

  /* may have to call this multiple times for breaks */
  do
  {
    (void) ClearCommError((HANDLE) OsHandle, &lErrors, &ComStat);
  } while (lErrors && --MaxTries);

  /* purge regardless */
  (void) PurgeComm((HANDLE) OsHandle, 
    (PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR));

  /* return a generic error.  
   * CONSIDER; We could parse and return better info.
   */

  return B_E_ERROR;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestDeviceConnect()
 * We set our DTR (card's DSR) to signal connect.
 * -------------------------------------------------------------------------- */

b_errtype BestSerDeviceConnect(b_portnumtype OsHandle)
{
  if(OsHandle == INVALID_OS_HANDLE)
    return B_E_BAD_HANDLE;

  /* start with a clean slate */
  (void) BestClearCommError(OsHandle);
  (void) EscapeCommFunction((HANDLE) OsHandle, SETDTR);
  DBG_PRINT("\nSerial Port Connected");

  return B_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestCheckConnection()
 * Card sets its DTR (our DSR) to signal connected.
 * -------------------------------------------------------------------------- */

b_errtype BestSerCheckConnection(b_portnumtype OsHandle)
{
  DWORD dwRegVal = 0; /* MS_DSR_ON; */

  if(OsHandle == INVALID_OS_HANDLE)
    return B_E_BAD_HANDLE;

  if( GetCommModemStatus((HANDLE) OsHandle, &dwRegVal) )
  {
    /* DSR high ? */
    if(MS_DSR_ON & dwRegVal)
      return B_E_OK;
  }
  else
  {
    /* This should NEVER happen (mixing baud rates will do it!)
     * Treat as a serious problem but DON'T disconnect.
     */
    dwRegVal = 0;
    DBG_ApiFailureIsZero(dwRegVal);
    BestSerOnReadOrWriteError(OsHandle);
  }

  return B_E_NOT_CONNECTED;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestReleaseConnection()
 * We clear our DTR (card's DSR) to signal disconnect.
 * -------------------------------------------------------------------------- */

void BestSerReleaseConnection(b_portnumtype OsHandle)
{
  if(OsHandle == INVALID_OS_HANDLE)
    return;

  (void) EscapeCommFunction((HANDLE) OsHandle, CLRDTR);
  DBG_PRINT("\nSerial Port Disconnected");
}


/* --------------------------------------------------------------------------
 * Should only be called from BestOnReadError()
 * -------------------------------------------------------------------------- */

b_errtype BestSerOnReadOrWriteError(b_portnumtype OsHandle)
{
#ifdef BEST_DEBUG

/* DEBUG ONLY ... display the real error */

  DCB dcb;
  COMMTIMEOUTS touts;
  unsigned long lErrors;
  COMSTAT ComStat;
  char buf[200];

  /* GetCommTimeouts() returns 0 on failure */
  if(GetCommTimeouts((HANDLE) OsHandle, &touts))
  {
    sprintf(buf, "\nCurrent Timeouts: %d %d %d %d %d", 
      touts.ReadIntervalTimeout,
      touts.ReadTotalTimeoutMultiplier,
      touts.ReadTotalTimeoutConstant,
      touts.WriteTotalTimeoutMultiplier,
      touts.WriteTotalTimeoutConstant
      );
  }
  else
  {
    strcpy(buf, "\n Failure getting Win API Current Timeouts");
  }

  DBG_PRINT(buf);

  /* can break here if required */
  (void) GetCommState((HANDLE) OsHandle, &dcb);

  (void) ClearCommError((HANDLE) OsHandle, &lErrors, &ComStat);

  if(lErrors)
  {
    if(lErrors & CE_RXOVER)   DBG_PRINT("\n Comm Err *** Receive Queue overflow");
    if(lErrors & CE_OVERRUN)  DBG_PRINT("\n Comm Err *** Receive Overrun Error");
    if(lErrors & CE_RXPARITY) DBG_PRINT("\n Comm Err *** Receive Parity Error");
    if(lErrors & CE_FRAME)    DBG_PRINT("\n Comm Err *** Receive Framing error");
    if(lErrors & CE_BREAK)    DBG_PRINT("\n Comm Err *** Break Detected");
    if(lErrors & CE_TXFULL)   DBG_PRINT("\n Comm Err *** TX Queue is full");
    if(lErrors & CE_TXFULL)   DBG_PRINT("\n Comm Err *** Requested mode unsupported");
  }

  /* COMSTAT structure contains information regarding
   * communications status.    
   */
  if (ComStat.fCtsHold)  DBG_PRINT("\n ComStat *** Tx waiting for CTS signal");
  if (ComStat.fDsrHold)  DBG_PRINT("\n ComStat *** Tx waiting for DSR signal");
  if (ComStat.fRlsdHold) DBG_PRINT("\n ComStat *** Tx waiting for RLSD signal");
  if (ComStat.fXoffHold) DBG_PRINT("\n ComStat *** Tx waiting, XOFF char rec'd");
  if (ComStat.fXoffSent) DBG_PRINT("\n ComStat *** Tx waiting, XOFF char sent");
  if (ComStat.fEof)      DBG_PRINT("\n ComStat *** EOF character received");
  if (ComStat.fTxim)     DBG_PRINT("\n ComStat *** Character waiting for Tx; char queued with TransmitCommChar");
  if (ComStat.cbInQue)   DBG_PRINT("\n ComStat *** byte(s) have been received, but not read");
  if (ComStat.cbOutQue)  DBG_PRINT("\n ComStat *** byte(s) are awaiting transfer");

  (void) PurgeComm((HANDLE) OsHandle, 
    (PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR));

  return B_E_OK;

#else

  if(OsHandle == INVALID_OS_HANDLE)
    return B_E_BAD_HANDLE;

  return BestClearCommError(OsHandle);

#endif
}


/* --------------------------------------------------------------------------
 * Should only be called from BestPortTimeoutSet()
 * -------------------------------------------------------------------------- */
b_errtype BestSerPortTimeoutSet(b_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  return SetCommTimeouts(OsHandle, (LPCOMMTIMEOUTS) pCallersTimeouts) ?
          B_E_OK : 
          B_E_SETTING_TIMEOUTS;
}


/******************************************************************************/

#define SER_IN_BUF_SIZE 4096
#define SER_OUT_BUF_SIZE 4096

b_portnumtype BestOpenCOMInitial(int num, b_int32 Baudrate)
{
  HANDLE OsHandle;
  DWORD brate;
  DCB dcb;

  LPCTSTR lpComPort;		/* pointer to com port string    */
  char comPortStr[6];		/* "COM" + double digit + '\0' ! */

  /* check for valid port number */
  if (num < 1 && num > 99)
  {
    return INVALID_OS_HANDLE;
  }

  sprintf (comPortStr, "COM%d", num);
  lpComPort = (LPCTSTR) comPortStr;

  /* open the comport as a file */
  if( INVALID_HANDLE_VALUE == 
      (OsHandle = CreateFile(lpComPort,
                GENERIC_READ | GENERIC_WRITE,
                0,                        /* no sharing */
                NULL,                     /* security attributes */
                OPEN_EXISTING,
                0,                        /* file attributes */
                NULL
    ))) return INVALID_OS_HANDLE;

  /* check the baudrate */
  switch (Baudrate)
  {
  case 9600:
    brate = CBR_9600;
    break;
  case 19200:
    brate = CBR_19200;
    break;
  case 38400:
    brate = CBR_38400;
    break;
  case 57600:
    brate = CBR_57600;
    break;
  case 115200:
    brate = CBR_115200;
    break;
  case 124800:
    brate = CBR_128000;
    break;
  default:
    BestCloseSerial(OsHandle);
    return INVALID_OS_HANDLE;
  }

  /* No monitoring */
  (void) SetCommMask(OsHandle, 0);

  /* buffer sizes */
  (void) SetupComm(OsHandle, SER_IN_BUF_SIZE, SER_OUT_BUF_SIZE);

  /* setup protocol and line control.
   * DTR (from PC) and DSR (from Card) are used to indicate "connected".
   * RTS/CTS used for hardware flow control.  
   * DSR sensitivity is used to help prevent junk in the input buffer.
   */
  (void) GetCommState(OsHandle, &dcb);

  dcb.DCBlength = sizeof(DCB);
  dcb.BaudRate = brate;
  dcb.fBinary = TRUE;
  dcb.fParity = FALSE;
  dcb.fOutxCtsFlow = TRUE;
  dcb.fOutxDsrFlow = TRUE;  /* was false */
  dcb.fDtrControl = DTR_CONTROL_DISABLE;
  dcb.fDsrSensitivity = TRUE;
  dcb.fOutX = FALSE;
  dcb.fInX = FALSE;
  dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;
  dcb.fAbortOnError = FALSE;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.StopBits = ONESTOPBIT;

  if (!SetCommState(OsHandle, &dcb))
  {
    BestCloseSerial(OsHandle);
    return INVALID_OS_HANDLE;
  }

  BestSerReleaseConnection((b_portnumtype) OsHandle);
  return (b_portnumtype) OsHandle;
}


/******************************************************************************/
void BestCloseSerial(b_portnumtype OsHandle)
{
  BestSerReleaseConnection(OsHandle);
  (void) CloseHandle((HANDLE) OsHandle);
}
